<?php

namespace App\Filament\Resources;

use App\Filament\Resources\PropertyCategoryResource\Pages;
use App\Filament\Resources\PropertyCategoryResource\RelationManagers;
use App\Models\PropertyCategory;
use Filament\Forms;
use Filament\Forms\Components\FileUpload;
use Filament\Forms\Components\RichEditor;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Columns\ToggleColumn;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Illuminate\Support\Str;

class PropertyCategoryResource extends Resource
{
    protected static ?string $model = PropertyCategory::class;

    protected static ?string $navigationIcon = 'heroicon-o-rectangle-stack';
      protected static ?string $navigationGroup = 'Property Management';

    protected static ?int $navigationSort = 2;

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Section::make()
                    ->schema([
                        TextInput::make('name')
                            ->live(debounce: 5000)
                            ->required()->minLength(1)->maxLength(150)
                            ->afterStateUpdated(function (string $operation, $state, Forms\Set $set) {
                                if ($operation === 'edit') {
                                    return;
                                }

                                $set('slug', Str::slug($state));
                            }),
                        TextInput::make('slug')->required()->readOnly()->minLength(1)->unique(ignoreRecord: true)->maxLength(150),
                        Select::make('property_type_id')
                            ->label('Property Type')
                            ->relationship('propertyTypes','name')
                            ->multiple()
                            ->preload()
                            ->native(false)->required(),
                        RichEditor::make('description')->label("Short Description"),
                    ])->columns(2),

                Section::make()
                    ->schema([
                        FileUpload::make('image')->disk('public')->label("Image Cover"),
                        Select::make('status')
                            ->options([
                                '1' => 'Published',
                                '0' => 'Draft'
                            ])->native(false)->required()
                    ])->columns(2),
                Section::make()
                    ->schema([
                        Textarea::make('meta_description')->nullable()->label('SEO Meta Description'),
                        Textarea::make('keywords')->nullable()->label('SEO Keywords'),


                    ])
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('name')->label(" Name"),
                ToggleColumn::make('status')->label("Status"),
            ])
            ->filters([
                //
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListPropertyCategories::route('/'),
            'create' => Pages\CreatePropertyCategory::route('/create'),
            'edit' => Pages\EditPropertyCategory::route('/{record}/edit'),
        ];
    }
}
