<?php

namespace App\Filament\Resources;

use App\Filament\Resources\BlogResource\Pages;
use App\Filament\Resources\BlogResource\RelationManagers;
use App\Models\Blog;
use Filament\Facades\Filament;
use Filament\Forms;
use Filament\Forms\Components\FileUpload;
use Filament\Forms\Components\RichEditor;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Hidden;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Columns\ImageColumn;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Columns\ToggleColumn;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;

class BlogResource extends Resource
{
    protected static ?string $model = Blog::class;

    protected static ?string $navigationIcon = 'heroicon-o-rectangle-stack';
    protected static ?string $navigationGroup = 'Blogs and Articles';

    protected static ?int $navigationSort = 2;

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Section::make('Blog Title & Description')
                    ->schema([
                      //  TextInput::make('user_id')->default(auth()->id()),
                        Hidden::make('user_id')->default(auth()->id()),


                        TextInput::make('name')
                            ->live(debounce: 5000)
                            ->required()->minLength(1)->maxLength(200)
                            ->afterStateUpdated(function (string $operation, $state, Forms\Set $set) {
                                if ($operation === 'edit') {
                                    return;
                                }

                                $set('slug', Str::slug($state));
                            })->columnSpanFull(),
                        TextInput::make('slug')->label('Slug URL: Automatically updates')
                            ->required()
                            ->readOnly()
                            ->minLength(1)
                            ->unique(ignoreRecord: true)
                            ->maxLength(200)
                            ->columnSpanFull(),
                        Select::make('blog_category_id')
                            ->label('Blog Category')

                            //->options(PropertyType::where('status',1)->pluck('name','id')->toArray())
                           ->relationship('blogCategory','name')
                            ->searchable()
                            ->preload()
                            ->native(false)
                            ->required(),
                        Select::make('status')
                            ->options([
                                '1' => 'Published',
                                '0' => 'Draft'
                            ])->native(false)->required(),

                        RichEditor::make('description')
                            ->required()
                            ->label("Blog Content")
                            ->columnSpanFull(),
                        FileUpload::make('image')->label("Image Cover")
                            ->required()
                            ->directory('blogs') ->columnSpanFull(),
                    ])->columns(2),
                Section::make('SEO Titles')
                    ->schema([
                        Textarea::make('meta_description')->nullable()->label('SEO Meta Description'),
                        Textarea::make('keywords')->nullable()->label('SEO Keywords'),


                    ])
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('name')->label(" Name"),
                ImageColumn::make('image')->label(" Image"),
                TextColumn::make('blogCategory.name')->label(" Blog Category"),
                TextColumn::make('user.name')->label(" Author"),
                ToggleColumn::make('status')->label("Status"),
                TextColumn::make('created_at')->label("Uploaded on") ->dateTime(),
            ])
            ->filters([
                //
            ])
            ->actions([
                Tables\Actions\ViewAction::make(),
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListBlogs::route('/'),
            'create' => Pages\CreateBlog::route('/create'),
            'view' => Pages\ViewBlog::route('/{record}'),
            'edit' => Pages\EditBlog::route('/{record}/edit'),
        ];
    }
}
